#!/usr/bin/env php
<?php
/**
 * order_printer.php — Poller de pedidos + impressão automática
 * ------------------------------------------------------------
 * • Roda em localhost (CLI do PHP) e consulta um MySQL remoto (seu servidor)
 * • Busca pedidos novos a cada X segundos
 * • Imprime automaticamente em impressora térmica 80mm (ESC/POS) via USB
 * • Marca o pedido como impresso para não duplicar
 *
 * Requisitos:
 *  - PHP 8.1+ com extensões: pdo_mysql, intl (opcional)
 *  - Composer + pacote mike42/escpos-php (para imprimir na USB)
 *  - Impressora instalada no sistema (Windows: fila do Windows; Linux: CUPS)
 *
 * Como usar (resumo):
 *  1) Salve este arquivo como order_printer.php
 *  2) Execute: php order_printer.php (ele vai gerar um config.ini na 1ª vez)
 *  3) Edite config.ini (DB, impressora, intervalo, etc.)
 *  4) Instale dependências: composer require mike42/escpos-php
 *  5) Execute novamente: php order_printer.php
 *  6) Para rodar em background: ver instruções no final do arquivo (Windows/Linux)
 */

const VERSION = '1.0.0';

date_default_timezone_set('America/Fortaleza');
mb_internal_encoding('UTF-8');

$BASE = __DIR__;
$CONFIG_FILE = $BASE . DIRECTORY_SEPARATOR . 'config.ini';
$LOG_FILE = $BASE . DIRECTORY_SEPARATOR . 'printer.log';

// ------------------------------------------------------------
// Bootstrap do config
// ------------------------------------------------------------
$defaultConfig = <<<INI
[app]
# Intervalo entre varreduras (segundos)
poll_interval = 5
# Quantos pedidos pegar por ciclo
default_batch_size = 5
# ID (opcional) para identificar este terminal impressor
terminal_id = "caixa-01"

[db]
host = "SEU_HOST_MYSQL"
port = 3306
schema = "SEU_SCHEMA"
user = "SEU_USUARIO"
pass = "SUA_SENHA"
# Ajuste o mapeamento de status que devem ser impressos
printable_statuses[] = "PAID"
printable_statuses[] = "CONFIRMED"
# Nome das tabelas/colunas (ajuste conforme seu banco)
orders_table = "orders"
order_items_table = "order_items"
# Colunas essenciais na tabela de pedidos (customize os nomes abaixo)
col_order_id = "id"
col_status = "status"
col_printed_at = "printed_at"
col_print_claimed_at = "print_claimed_at"
col_print_claimed_by = "print_claimed_by"
col_created_at = "created_at"
col_total = "total"
col_table_no = "table_no"
col_customer = "customer_name"
col_notes = "notes"

# Colunas essenciais na tabela de itens
col_item_order_id = "order_id"
col_item_name = "name"
col_item_qty = "qty"
col_item_unit_price = "unit_price"

[printer]
# Tipo de conector: windows|cups|network|file
connector = "windows"
# Nome da impressora (Windows: nome da fila; Linux CUPS: queue name; Network: ip:port)
printer_name = "NOME_DA_IMPRESSORA"
# Para connector=file, caminho do arquivo de saída de teste
output_file = "ticket.txt"
# Charset da impressora (opcional): CP860 (Português), CP437, CP850...
charset = "CP860"
# Largura (caracteres) da bobina (80mm costuma 48; algumas usam 42)
chars_per_line = 48
# Cabeçalho do cupom
header_line1 = "RESTAURANTE EXEMPLO"
header_line2 = "CNPJ 00.000.000/0001-00"
header_line3 = "Rua Tal, 123 - Centro"
# Rodapé do cupom
footer_line1 = "Obrigado pela preferência!"
footer_line2 = "Siga @restaurante"
# Opcional: imprimir QR com ID do pedido
print_qr = true

INI;

if (!file_exists($CONFIG_FILE)) {
    file_put_contents($CONFIG_FILE, $defaultConfig);
    echo "\n[order_printer] Primeira execução.\n";
    echo "Foi criado um config.ini em: {$CONFIG_FILE}\n";
    echo "Edite as credenciais do banco e o nome da impressora, depois rode de novo.\n";
    exit(0);
}

$config = parse_ini_file($CONFIG_FILE, true, INI_SCANNER_TYPED);
if (!$config) {
    fwrite(STDERR, "Erro ao ler config.ini\n");
    exit(1);
}

// ------------------------------------------------------------
// Dependências de impressão (mike42/escpos-php)
// ------------------------------------------------------------
$HAS_ESC_POS = false;
$AUTOLOAD = $BASE . '/vendor/autoload.php';
if (file_exists($AUTOLOAD)) {
    require_once $AUTOLOAD;
    $HAS_ESC_POS = true;
}

// Tipos importados apenas se o pacote existir
if ($HAS_ESC_POS) {
    //use Mike42\Escpos\Printer;
    //use Mike42\Escpos\PrintConnectors\WindowsPrintConnector;
    //use Mike42\Escpos\PrintConnectors\CupsPrintConnector;
    //use Mike42\Escpos\PrintConnectors\NetworkPrintConnector;
    //use Mike42\Escpos\PrintConnectors\FilePrintConnector;
}

// ------------------------------------------------------------
// Utilitários
// ------------------------------------------------------------
function logmsg(string $msg) {
    global $LOG_FILE;
    $line = sprintf("[%s] %s\n", date('Y-m-d H:i:s'), $msg);
    file_put_contents($LOG_FILE, $line, FILE_APPEND);
}

function pdo(array $cfg): PDO {
    static $pdo;
    if ($pdo) return $pdo;
    $dsn = sprintf("mysql:host=%s;port=%d;dbname=%s;charset=utf8mb4",
        $cfg['host'], (int)$cfg['port'], $cfg['schema']
    );
    $pdo = new PDO($dsn, $cfg['user'], $cfg['pass'], [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_PERSISTENT => true,
    ]);
    return $pdo;
}

function arr_get(array $a, string $k, $default = null) { return $a[$k] ?? $default; }

function within(array $row, array $keys): bool {
    foreach ($keys as $k) if (!array_key_exists($k, $row)) return false;
    return true;
}

function str_pad_right(string $s, int $w): string { return mb_strwidth($s) >= $w ? $s : $s . str_repeat(' ', $w - mb_strwidth($s)); }
function str_pad_left(string $s, int $w): string { return mb_strwidth($s) >= $w ? $s : str_repeat(' ', $w - mb_strwidth($s)) . $s; }

// Alinha duas colunas em uma linha fixa
function two_cols(string $left, string $right, int $width): string {
    $lenL = mb_strwidth($left);
    $lenR = mb_strwidth($right);
    $spaces = max(1, $width - $lenL - $lenR);
    return $left . str_repeat(' ', $spaces) . $right;
}

// Quebra esperta por palavras
function wrap_text(string $text, int $width): array {
    $words = preg_split('/\s+/', trim($text));
    $lines = [];
    $line = '';
    foreach ($words as $w) {
        $trial = trim($line === '' ? $w : ($line . ' ' . $w));
        if (mb_strwidth($trial) <= $width) {
            $line = $trial;
        } else {
            if ($line !== '') $lines[] = $line;
            $line = $w;
        }
    }
    if ($line !== '') $lines[] = $line;
    return $lines;
}

// ------------------------------------------------------------
// Claim de pedidos p/ evitar duplicidade
// ------------------------------------------------------------
function claimOrders(PDO $pdo, array $cfgDb, array $cfgApp, int $limit): array {
    $t = $cfgDb['orders_table'];
    $colId = $cfgDb['col_order_id'];
    $colStatus = $cfgDb['col_status'];
    $colPrintedAt = $cfgDb['col_printed_at'];
    $colClaimAt = $cfgDb['col_print_claimed_at'];
    $colClaimBy = $cfgDb['col_print_claimed_by'];
    $colCreated = $cfgDb['col_created_at'];
    $statuses = arr_get($cfgDb, 'printable_statuses', []);

    if (!$statuses) { $statuses = ['PAID','CONFIRMED']; }
    $in = implode(',', array_fill(0, count($statuses), '?'));

    // Limpa claims antigos (timeout 10 minutos)
    $pdo->prepare("UPDATE `$t` SET `$colClaimAt` = NULL, `$colClaimBy` = NULL WHERE `$colPrintedAt` IS NULL AND `$colClaimAt` < (NOW() - INTERVAL 10 MINUTE)")->execute();

    // Efetua claim
    $claimer = gethostname() . ':' . getmypid() . ':' . arr_get($cfgApp, 'terminal_id', 'terminal');
    $sqlClaim = "UPDATE `$t` SET `$colClaimAt` = NOW(), `$colClaimBy` = ?
                 WHERE `$colPrintedAt` IS NULL AND (`$colClaimAt` IS NULL)
                   AND `$colStatus` IN ($in)
                 ORDER BY `$colCreated` ASC LIMIT $limit";
    $stmt = $pdo->prepare($sqlClaim);
    $stmt->execute(array_merge([$claimer], $statuses));

    // Busca os pedidos que foram claimados por este terminal agora
    $sqlSel = "SELECT * FROM `$t` WHERE `$colPrintedAt` IS NULL AND `$colClaimBy` = ? ORDER BY `$colCreated` ASC LIMIT $limit";
    $rows = $pdo->prepare($sqlSel);
    $rows->execute([$claimer]);
    return $rows->fetchAll();
}

function fetchItems(PDO $pdo, array $cfgDb, $orderId): array {
    $ti = $cfgDb['order_items_table'];
    $colItemOrderId = $cfgDb['col_item_order_id'];
    $sql = "SELECT * FROM `$ti` WHERE `$colItemOrderId` = ? ORDER BY 1";
    $st = $pdo->prepare($sql);
    $st->execute([$orderId]);
    return $st->fetchAll();
}

function markPrinted(PDO $pdo, array $cfgDb, $orderId): void {
    $t = $cfgDb['orders_table'];
    $colId = $cfgDb['col_order_id'];
    $colPrintedAt = $cfgDb['col_printed_at'];
    $colClaimAt = $cfgDb['col_print_claimed_at'];
    $colClaimBy = $cfgDb['col_print_claimed_by'];
    $pdo->prepare("UPDATE `$t` SET `$colPrintedAt` = NOW(), `$colClaimAt` = NULL, `$colClaimBy` = NULL WHERE `$colId` = ?")
        ->execute([$orderId]);
}

// ------------------------------------------------------------
// Impressão
// ------------------------------------------------------------
function openPrinter(array $cfgPrinter) {
    global $HAS_ESC_POS;
    if (!$HAS_ESC_POS) throw new RuntimeException("Pacote mike42/escpos-php não encontrado. Rode 'composer require mike42/escpos-php'.");

    $connector = strtolower($cfgPrinter['connector'] ?? 'windows');
    $name = $cfgPrinter['printer_name'] ?? '';

    switch ($connector) {
        case 'windows':
            return new Mike42\Escpos\Printer(new Mike42\Escpos\PrintConnectors\WindowsPrintConnector($name));
        case 'cups':
            return new Mike42\Escpos\Printer(new Mike42\Escpos\PrintConnectors\CupsPrintConnector($name));
        case 'network':
            // Ex: 192.168.0.50:9100
            [$host,$port] = array_pad(explode(':', $name, 2), 2, 9100);
            return new Mike42\Escpos\Printer(new Mike42\Escpos\PrintConnectors\NetworkPrintConnector($host, (int)$port));
        case 'file':
            $file = $cfgPrinter['output_file'] ?? 'ticket.txt';
            return new Mike42\Escpos\Printer(new Mike42\Escpos\PrintConnectors\FilePrintConnector($file));
        default:
            throw new InvalidArgumentException("Connector desconhecido: $connector");
    }
}

function formatMoney($v): string { return 'R$ ' . number_format((float)$v, 2, ',', '.'); }

function printOrder(array $order, array $items, array $cfg): void {
    $pCfg = $cfg['printer'];
    $w = (int)($pCfg['chars_per_line'] ?? 48);

    $printer = openPrinter($pCfg);

    // Charset
    if (!empty($pCfg['charset'])) {
        try { $printer->selectCharacterTable($pCfg['charset']); } catch (Throwable $e) {}
    }

    // Header
    $headerLines = [];
    foreach (['header_line1','header_line2','header_line3'] as $h) {
        $val = trim((string)arr_get($pCfg, $h, ''));
        if ($val !== '') $headerLines[] = $val;
    }

    $printer->setJustification(Mike42\Escpos\Printer::JUSTIFY_CENTER);
    $printer->setEmphasis(true);
    foreach ($headerLines as $hl) {
        foreach (wrap_text($hl, $w) as $ln) $printer->text($ln . "\n");
    }
    $printer->setEmphasis(false);
    $printer->text(str_repeat('-', $w) . "\n");

    // Info do pedido
    $id = $order['id'] ?? $order[arr_get($cfg['db'], 'col_order_id', 'id')] ?? 'N/A';
    $status = $order[arr_get($cfg['db'], 'col_status', 'status')] ?? '';
    $tableNo = $order[arr_get($cfg['db'], 'col_table_no', 'table_no')] ?? '';
    $customer = $order[arr_get($cfg['db'], 'col_customer', 'customer_name')] ?? '';
    $created = $order[arr_get($cfg['db'], 'col_created_at', 'created_at')] ?? date('Y-m-d H:i:s');

    $printer->setJustification(Mike42\Escpos\Printer::JUSTIFY_LEFT);
    $printer->setEmphasis(true);
    $printer->text(two_cols("PEDIDO #$id", date('d/m/Y H:i', strtotime($created)), $w) . "\n");
    $printer->setEmphasis(false);
    if ($tableNo !== '') $printer->text("Mesa: {$tableNo}\n");
    if ($customer !== '') $printer->text("Cliente: {$customer}\n");
    if ($status !== '') $printer->text("Status: {$status}\n");
    $printer->text(str_repeat('-', $w) . "\n");

    // Itens
    foreach ($items as $it) {
        $name = (string)arr_get($it, arr_get($cfg['db'], 'col_item_name', 'name'), 'Item');
        $qty = (float)arr_get($it, arr_get($cfg['db'], 'col_item_qty', 'qty'), 1);
        $unit = (float)arr_get($it, arr_get($cfg['db'], 'col_item_unit_price', 'unit_price'), 0);
        $total = $qty * $unit;

        $lineTop = two_cols(sprintf('%sx %s', rtrim(rtrim(number_format($qty,2,',','.'), '0'),','), $name), formatMoney($total), $w);
        $printer->text($lineTop . "\n");
    }

    $printer->text(str_repeat('-', $w) . "\n");

    // Totais
    $totalOrder = $order[arr_get($cfg['db'], 'col_total', 'total')] ?? null;
    if ($totalOrder !== null) {
        $printer->setEmphasis(true);
        $printer->text(two_cols('TOTAL', formatMoney($totalOrder), $w) . "\n");
        $printer->setEmphasis(false);
    }

    // Observações
    $notes = (string)($order[arr_get($cfg['db'], 'col_notes', 'notes')] ?? '');
    if (trim($notes) !== '') {
        $printer->text(str_repeat('-', $w) . "\n");
        $printer->setEmphasis(true);
        $printer->text("Observações:\n");
        $printer->setEmphasis(false);
        foreach (wrap_text($notes, $w) as $ln) $printer->text($ln . "\n");
    }

    $printer->text(str_repeat('-', $w) . "\n");

    // QR opcional
    if (!empty($pCfg['print_qr'])) {
        try { $printer->qrCode((string)$id, Mike42\Escpos\Printer::QR_ECLEVEL_L, 6); $printer->text("\n"); } catch (Throwable $e) {}
    }

    // Rodapé
    $printer->setJustification(Mike42\Escpos\Printer::JUSTIFY_CENTER);
    foreach (['footer_line1','footer_line2'] as $f) {
        $val = trim((string)arr_get($pCfg, $f, ''));
        if ($val !== '') foreach (wrap_text($val, $w) as $ln) $printer->text($ln . "\n");
    }

    // Corte e bip
    try { $printer->cut(); } catch (Throwable $e) {}
    try { $printer->pulse(); } catch (Throwable $e) {}

    $printer->close();
}

// ------------------------------------------------------------
// Loop principal
// ------------------------------------------------------------
$poll = max(1, (int)arr_get($config['app'], 'poll_interval', 5));
$batch = max(1, (int)arr_get($config['app'], 'default_batch_size', 5));

logmsg("Iniciando order_printer v" . VERSION . " | intervalo={$poll}s batch={$batch}");

while (true) {
    try {
        $pdo = pdo($config['db']);
        $orders = claimOrders($pdo, $config['db'], $config['app'], $batch);
        if ($orders) logmsg("Encontrados ".count($orders)." pedido(s) para imprimir");
        foreach ($orders as $o) {
            try {
                $orderId = $o[$config['db']['col_order_id']];
                $items = fetchItems($pdo, $config['db'], $orderId);
                printOrder($o, $items, $config);
                markPrinted($pdo, $config['db'], $orderId);
                logmsg("Pedido #{$orderId} impresso com sucesso.");
            } catch (Throwable $e) {
                logmsg("ERRO ao imprimir pedido: " . $e->getMessage());
                // Solta o claim para tentar novamente depois
                try {
                    $t = $config['db']['orders_table'];
                    $colId = $config['db']['col_order_id'];
                    $pdo->prepare("UPDATE `$t` SET `{$config['db']['col_print_claimed_at']}`=NULL, `{$config['db']['col_print_claimed_by']}`=NULL WHERE `$colId`=?")
                        ->execute([$o[$config['db']['col_order_id']]]);
                } catch (Throwable $e2) { logmsg("ERRO limpando claim: ".$e2->getMessage()); }
            }
        }
    } catch (Throwable $e) {
        logmsg("Falha no ciclo: " . $e->getMessage());
    }

    sleep($poll);
}

// ------------------------------------------------------------
// MIGRAÇÕES SQL (ajuste conforme seu banco) — EXEMPLO
// ------------------------------------------------------------
/*
ALTER TABLE orders
  ADD COLUMN printed_at DATETIME NULL,
  ADD COLUMN print_claimed_at DATETIME NULL,
  ADD COLUMN print_claimed_by VARCHAR(100) NULL,
  ADD INDEX idx_orders_print (printed_at, print_claimed_at);

-- Exemplo de estrutura mínima
-- CREATE TABLE orders (
--   id BIGINT PRIMARY KEY AUTO_INCREMENT,
--   status VARCHAR(20) NOT NULL,
--   customer_name VARCHAR(120) NULL,
--   table_no VARCHAR(10) NULL,
--   total DECIMAL(10,2) NOT NULL DEFAULT 0,
--   notes TEXT NULL,
--   created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
--   printed_at DATETIME NULL,
--   print_claimed_at DATETIME NULL,
--   print_claimed_by VARCHAR(100) NULL
-- ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
--
-- CREATE TABLE order_items (
--   id BIGINT PRIMARY KEY AUTO_INCREMENT,
--   order_id BIGINT NOT NULL,
--   name VARCHAR(255) NOT NULL,
--   qty DECIMAL(10,2) NOT NULL DEFAULT 1,
--   unit_price DECIMAL(10,2) NOT NULL DEFAULT 0,
--   FOREIGN KEY (order_id) REFERENCES orders(id)
-- ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
*/

// ------------------------------------------------------------
// RODAR EM BACKGROUND (resumo)
// ------------------------------------------------------------
/*
Windows (recomendado Task Scheduler):
1) Instale a impressora e confirme o NOME exato da fila (Ex.: "MP-4200 TH").
2) No diretório do script, rode:  composer require mike42/escpos-php
3) Edite config.ini (db + printer_name + connector=windows).
4) Teste: php order_printer.php (verifique printer.log)
5) Agende:
   - Abra Agendador de Tarefas > Criar Tarefa
   - Disparador: Ao iniciar o Windows (ou Ao fazer logon)
   - Ação: Programa: C:\\php\\php.exe
            Argumentos: C:\\caminho\\order_printer.php
            Iniciar em: C:\\caminho\\
   - Marque "Executar com privilégios mais altos" e "Executar esteja o usuário logado ou não".

Linux (systemd):
1) Instale CUPS e configure a impressora. Em config.ini: connector=cups, printer_name=nome_da_fila
2) Unit file /etc/systemd/system/order-printer.service
   [Unit]
   Description=Order Printer PHP
   After=network.target

   [Service]
   WorkingDirectory=/opt/order-printer
   ExecStart=/usr/bin/php /opt/order-printer/order_printer.php
   Restart=always
   RestartSec=3
   User=www-data

   [Install]
   WantedBy=multi-user.target

   Comandos:
   sudo systemctl daemon-reload
   sudo systemctl enable --now order-printer

Rede (impressora Ethernet):
- Em [printer], use connector=network e printer_name=IP:9100

Teste sem impressora:
- Em [printer], use connector=file e output_file=ticket.txt
*/
